// Type definitions for HistropediaJS – public API only

// UMD global support
export as namespace Histropedia;

// ------------------------------------------------------------------------------------
// Helpers
// ------------------------------------------------------------------------------------

/**
 * Deeply partial helper for ergonomic option updates.
 *
 * Notes:
 * - Functions are left as-is (not recursed).
 * - Arrays/tuples are treated as replace-whole values (not recursed).
 */
export type DeepPartial<T> =
  T extends (...args: any) => any ? T :
  T extends readonly any[] ? T :
  T extends object ? { [K in keyof T]?: DeepPartial<T[K]> } :
  T;

// ------------------------------------------------------------------------------------
// Date and precision
// ------------------------------------------------------------------------------------

/** Numeric precision constants (public). */
export const PRECISION_DAY: 11;
export const PRECISION_MONTH: 10;
export const PRECISION_YEAR: 9;
export const PRECISION_DECADE: 8;
export const PRECISION_CENTURY: 7;
export const PRECISION_MILLENNIUM: 6;
export const PRECISION_MILLION_YEARS: 3;
export const PRECISION_BILLION_YEARS: 1;

/**
 * Date precision used by Article/TimeBand ranges.
 * Matches the numeric constants exported above.
 */
export type Precision =
  | typeof PRECISION_DAY
  | typeof PRECISION_MONTH
  | typeof PRECISION_YEAR
  | typeof PRECISION_DECADE
  | typeof PRECISION_CENTURY
  | typeof PRECISION_MILLENNIUM
  | typeof PRECISION_MILLION_YEARS
  | typeof PRECISION_BILLION_YEARS;

/** Plain YMD date parts (no precision). */
export type DateParts = {
  year: number;
  month?: number;
  day?: number;
};

/** General date input that can include precision. */
export type DateInput = DateParts & { precision?: Precision };

/** Internal date input shape (consumer-friendly). */
export interface Dmy {
  year: number;
  month?: number;
  day?: number;
  precision?: Precision;
}

/** Formatting options for Dmy.format. */
export interface DmyFormatOptions {
  yearPrefix?: { value?: number; label?: string };
  bceText?: string;
  thousandsSeparator?: string;
}

/** Histropedia date helper instance. */
export interface DmyInstance extends Dmy {
  getNextDay(): DmyInstance;
  addDays(amount: number): DmyInstance;
  addMonths(amount: number): DmyInstance;
  addYears(amount: number): DmyInstance;
  getPreviousDay(): DmyInstance;
  format(format?: string, options?: DmyFormatOptions): string;
  isSame(other: Dmy): boolean;
  isAfter(other: Dmy): boolean;
  isBetween(from: Dmy, to: Dmy): boolean;
}

export interface DmyConstructor {
  new (year: number, month?: number, day?: number): DmyInstance;
  new (value: Dmy): DmyInstance;
  readonly prototype: DmyInstance;
  getByDayOfYear(year: number, dayOfYear: number): DmyInstance;
  now(): DmyInstance;
  fromString(dateString: string): DmyInstance;
}

export const Dmy: DmyConstructor;

// ------------------------------------------------------------------------------------
// Public constants (used in options)
// ------------------------------------------------------------------------------------

export const DENSITY_ALL: 0;
export const DENSITY_LOW: 1;
export const DENSITY_MEDIUM: 2;
export const DENSITY_HIGH: 3;
export type Density =
  | typeof DENSITY_ALL
  | typeof DENSITY_LOW
  | typeof DENSITY_MEDIUM
  | typeof DENSITY_HIGH;

export const RANGE_ALL: 0;
export const RANGE_SCREEN: 1;
export type AutoStackRange = typeof RANGE_ALL | typeof RANGE_SCREEN;

// ------------------------------------------------------------------------------------
// Animation types
// ------------------------------------------------------------------------------------

/** Allowed easing names used by animations. */
export type EasingName =
  | 'linear'
  | 'swing'
  | 'easeInQuad'
  | 'easeOutQuad'
  | 'easeInOutQuad'
  | 'easeInCubic'
  | 'easeOutCubic'
  | 'easeInOutCubic';

// ------------------------------------------------------------------------------------
// Style types
// ------------------------------------------------------------------------------------

export interface ArticleStyleText {
  font?: string;
  color?: string;
  align?: 'left' | 'right' | 'center' | 'start' | 'end';
  baseline?: 'top' | 'middle' | 'alphabetic' | 'bottom';
  margin?: number;
  lineHeight?: number;
  numberOfLines?: number;
  offsetY?: number;
}

export interface ArticleStyle {
  /** @defaultValue 180 */
  width?: number;
  /** Landscape card layout only. @defaultValue 70 */
  height?: number;
  /** Card background fill color. @defaultValue '#e9e9e9' */
  color?: string;
  /** Inner card background. @defaultValue '#fff' */
  backgroundColor?: string;
  /** Portrait card layout only. @defaultValue 3 */
  topRadius?: number;
  /** Landscape card layout only. @defaultValue 4 */
  borderRadius?: number;
  /**
   * Shorthand for the article image maximum height.
   *
   * @deprecated Deprecated since 1.3.0 — Use `image.maxHeight` instead.
   * @defaultValue 400
   */
  maxImageHeight?: number;
  image?: {
    /** @defaultValue 'natural' */
    shape?: 'natural' | 'square' | 'circle';
    /** @defaultValue 0 */
    margin?: number;
    /** @defaultValue 400 */
    maxHeight?: number;
    /** @defaultValue 0 */
    borderRadius?: number;
  };
  header?: {
    /** Portrait card layout only. @defaultValue 50 */
    height?: number;
    text?: ArticleStyleText;
  };
  subheader?: {
    /** Portrait card layout only. @defaultValue 30 */
    height?: number;
    /** Portrait card layout only. @defaultValue '#555' */
    color?: string;
    text?: ArticleStyleText;
  };
  shadow?: {
    /** @defaultValue 0 */ x?: number;
    /** @defaultValue 0 */ y?: number;
    /** @defaultValue 0 */ amount?: number;
    /** @defaultValue '#000' */ color?: string;
  };
  border?: {
    /** @defaultValue '#ddd' */ color?: string;
    /** @defaultValue 1 */ width?: number;
  };
  connectorLine?: {
    /** @defaultValue true */ visible?: boolean;
    /** @defaultValue 18 */ offsetX?: number;
    /** @defaultValue -5 */ offsetY?: number;
    /** @defaultValue 1 */ thickness?: number;
    arrow?: { width?: number; height?: number };
  };
  star?: {
    /** @defaultValue 16 */ width?: number;
    /** @defaultValue 3 */ margin?: number;
  };
}

export interface TimeBandStyleText {
  font?: string;
  color?: string;
  align?: 'left' | 'right' | 'center' | 'start' | 'end';
  baseline?: 'top' | 'middle' | 'alphabetic' | 'bottom';
  /** Vertical align inside band area. */
  verticalAlign?: 'top' | 'middle' | 'bottom';
  /** Horizontal/vertical padding in px. @defaultValue 4 */
  margin?: number;
  /** Extra vertical offset in px. @defaultValue 0 */
  offsetY?: number;
}

export interface TimeBandStyle {
  /** @defaultValue 'rgba(200, 200, 200, 0.25)' */
  backgroundColor?: string;
  border?: { color?: string; width?: number };
  text?: TimeBandStyleText;
}

export interface TimelineStyleMarkerKind {
  /** @defaultValue 12 (minor) / 30 (major) */ height?: number;
  /** Extend marker to canvas top/bottom. @defaultValue false */ extend?: boolean;
  color?: string;
  futureColor?: string;
}

export interface TimelineStyleOptions {
  mainLine?: {
    /** @defaultValue true */ visible?: boolean;
    /** @defaultValue 'bottom' */ position?: 'top' | 'bottom';
    /** thickness in px. @defaultValue 8 */ size?: number;
    /** Draw small article position ticks. @defaultValue true */
    showDateIndicators?: boolean;
  };
  draggingHighlight?: {
    /** @defaultValue true */ visible?: boolean;
    /** @defaultValue 'rgba(237, 247, 255, 0.5)' */ color?: string;
    /** Up/down extents relative to the main line. Accepts px, `'edge'`, or `{ value: 'edge', padding?: number }`.
    * @defaultValue {up: 0, down: 'edge'}
    */
    area?: { up?: AreaValue; down?: AreaValue };
  };
  marker?: {
    minor?: TimelineStyleMarkerKind;
    major?: TimelineStyleMarkerKind;
  };
  dateLabel?: {
    minor?: {
      font?: string;
      color?: string;
      futureColor?: string;
      textAlign?: CanvasTextAlign;
      offset?: { x?: number; y?: number };
      /** Text appended for BCE labeling. */
      bceText?: string;
      /** Thousands separator for large year labels. */
      thousandsSeparator?: string;
      /** Map of year-scale prefixes to display (e.g. ka, Ma, Ga). */
      yearPrefixes?: Record<string, { label: string; value: number; minDivision: number }>;
    };
    major?: {
      font?: string;
      color?: string;
      futureColor?: string;
      textAlign?: CanvasTextAlign;
      offset?: { x?: number; y?: number };
      bceText?: string;
      thousandsSeparator?: string;
      yearPrefixes?: Record<string, { label: string; value: number; minDivision: number }>;
    };
  };
  cursor?: {
    clickable?: string;
    timeline?: { hover?: string; dragging?: string };
    article?: { hoverDraggable?: string; dragging?: string };
  };
}

// ------------------------------------------------------------------------------------
// Data schemas for loading
// ------------------------------------------------------------------------------------

export interface ArticleData {
  id: string | number;
  title: string;
  subtitle?: string;
  from: DateInput;
  to?: DateInput;
  /** If true, the period ends at the present day. */
  isToPresent?: boolean;
  imageUrl?: string;
  rank?: number;
  starred?: boolean;
  hiddenByFilter?: boolean | ((article: Article) => boolean);
  hidePeriodLine?: boolean | ((article: Article) => boolean);
  /** Initial offset from computed position (px). */
  offsetLeft?: number;
  /** Initial offset from computed position (px). */
  offsetTop?: number;
  /** Override default card layout for this article. */
  cardLayout?: 'portrait' | 'landscape' | string;
  style?: ArticleStyle;
  hoverStyle?: ArticleStyle;
  activeStyle?: ArticleStyle;
}

export interface TimeBandData {
  id: string | number;
  title: string;
  from: DateInput;
  to?: DateInput;
  /** If true, the band ends at the present day. */
  isToPresent?: boolean;
  /** Visibility of this band: 'visible' | 'hidden' or a function returning one. Defaults to 'visible'. */
  visibility?: 'visible' | 'hidden' | ((band: TimeBand) => 'visible' | 'hidden');
  style?: TimeBandStyle;
}

// ------------------------------------------------------------------------------------
// Timeline options
// ------------------------------------------------------------------------------------

export interface TimelineZoomOptions {
  /** @defaultValue 34 */ initial?: number;
  /** @defaultValue 0 */ minimum?: number;
  /** @defaultValue 123 */ maximum?: number;
  /** Zoom step multiplier per wheel unit. @defaultValue 0.1 */
  wheelStep?: number;
  /** Overall wheel speed. @defaultValue 3 */
  wheelSpeed?: number;
  /** Allow Ctrl+wheel zoom. @defaultValue false */
  allowCtrlWheel?: boolean;
  /** 'auto' picks smooth vs discrete from device; 'discrete' forces stepped. @defaultValue 'auto' */
  wheelMode?: 'auto' | 'discrete';
  /** Gain for proportional (trackpad) mode. @defaultValue 2 */
  proportionalGain?: number;
  /** Exponent for proportional (trackpad) mode. @defaultValue 1.1 */
  proportionalExponent?: number;
  /** Zoom ratio between unit-size levels. @defaultValue 0.8 */
  ratio?: number;
  unitSize?: {
    /** Size of a day at zoom=0 (and density region size). @defaultValue 200 */
    initial?: number;
    /** Unit-size threshold to show minor labels. @defaultValue 48 */
    showMinorLabels?: number;
    /** Minimum unit size in px. @defaultValue 8 */
    minimum?: number;
  };
}

export interface ArticleOptions {
  /** Density filter for visible articles. @defaultValue DENSITY_ALL */
  density?: Density;
  /** Allow dragging event cards. @defaultValue true */
  draggable?: boolean;
  /** Distance of cards from main line in px. @defaultValue 350 */
  distanceToMainLine?: number;
  /** Collect ongoing events on main line. @defaultValue false */
  collectOngoing?: boolean;
  /** Auto stacking options. */
  autoStacking?: {
    /** @defaultValue true */
    active?: boolean;
    /** Distance between rows in px. @defaultValue 50 */
    rowSpacing?: number;
    /** Range considered for stacking. @defaultValue RANGE_ALL */
    range?: AutoStackRange;
    /** Fit to available height by adjusting rowSpacing. @defaultValue true */
    fitToHeight?: boolean;
    /** Top gap in px or a function returning px. @defaultValue 10 */
    topGap?: number | ((timeline: Timeline) => number);
  };
  periodLine?: {
    /** Gap above/below main line in px. @defaultValue 4 */
    spacing?: number;
    /** Line thickness in px. @defaultValue 10 */
    thickness?: number;
    stacking?: {
      /** Function name exported by sorters; defaults internally. */
      sorter?: any;
      /** Reverse stacking order. @defaultValue false */
      reverseOrder?: boolean;
    };
  };
  animation?: {
    fade?: { active?: boolean; duration?: number; easing?: EasingName };
    move?: { active?: boolean; duration?: number; easing?: EasingName };
  };
  star?: { /** Show star icon. @defaultValue true */ visible?: boolean };
  /** Default data merged into every loaded article. */
  defaultData?: Partial<ArticleData>;
  /** Default card layout name. @defaultValue 'portrait' */
  defaultCardLayout?: 'portrait' | 'landscape' | string;
  /** Base styles. */
  defaultStyle?: ArticleStyle;
  defaultHoverStyle?: ArticleStyle;
  defaultActiveStyle?: ArticleStyle;
  /** Per-layout style overrides registered via Timeline.registerCardLayout. */
  layoutStyles?: Record<string, Partial<{
    style: ArticleStyle;
    hoverStyle: ArticleStyle;
    activeStyle: ArticleStyle;
  }>>;
}

/** Pixels from the main line, the `'edge'` keyword, or `{ value: 'edge', padding?: number }` for edge-stretching with padding. */
export type AreaValue =
  | number
  | 'edge'
  | { value: 'edge'; padding?: number };

export interface TimeBandOptions {
  /** Enable drawing of bands. @defaultValue true */
  visible?: boolean;
  /** Reserve extra space near main line when bands are present. @defaultValue true */
  reserveSpace?: boolean;
  /** Pixels to reserve when reserveSpace is true. @defaultValue 20 */
  reserveSpacePixels?: number;
  /** Up/down extents of the band area relative to the main line. Accepts px, `'edge'`, or `{ value: 'edge', padding?: number }`. @defaultValue {up: 0, down: 'edge'} */
  area?: { up?: AreaValue; down?: AreaValue };
  /** Global default style merged into each band. */
  defaultStyle?: TimeBandStyle;
  /** Bands to load initially. */
  data?: TimeBandData[];
}

/**
 * Image loading and URL sanitization options.
 *
 * By default, the built-in sanitizer allows http, https, data:image/* and blob:.
 * You can restrict schemes and/or remote origins, or provide a custom sanitizer.
 */
export type ImageDecodeMode =
  | 'auto'
  | 'prefer-bitmap'
  | 'bitmap'
  | 'prefer-img'
  | 'img';

export interface ImageSanitizerOptions {
  /**
   * Allowed URL schemes for images.
   * @defaultValue ['http', 'https', 'data', 'blob']
   */
  allowedSchemes?: string[];

  /**
   * Allowed origins/hosts for remote images. Supports full origins, bare hostnames,
   * leading-dot, and wildcard-prefixed (`*.example.com`) entries.
   */
  allowedOrigins?: string[];
}

export interface ImageOptions {
  /**
   * Built-in URL sanitization controls.
   */
  sanitizer?: ImageSanitizerOptions;

  /**
   * Optional override for image URL sanitization. Receives the raw URL and the
   * current sanitizer configuration from `image.sanitizer`. Return a trusted URL
   * string to allow the load or null to block it.
   */
  customSanitizer?: (url: string, options: ImageSanitizerOptions) => string | null;

  /**
   * Maximum number of concurrent image fetch/decode operations. Values < 1 are clamped to 1.
   * @defaultValue 6
   */
  maxConcurrent?: number;

  /**
   * Maximum decoded raster memory (in bytes) retained by the image cache.
   * Set to `Infinity` to disable eviction, or reduce to bound memory usage.
   * @defaultValue 64 * 1024 * 1024 (64 MiB)
   */
  maxCacheBytes?: number;

  /**
   * Force cross-origin requests to opt-in via CORS (sets `crossOrigin = 'anonymous'`).
   * When true, images served without CORS headers will fail to load but canvas readbacks
   * remain untainted.
   * @defaultValue false
   */
  requireCORS?: boolean;

  /**
   * Controls how raster data is produced once an image loads.
   *
   * - 'auto' (default) prefers ImageBitmap when supported, falling back to `<img>`.
   * - 'prefer-bitmap' aggressively tries ImageBitmap but falls back to `<img>` if needed.
   * - 'bitmap' requires ImageBitmap support and treats failures as errors.
   * - 'prefer-img' keeps `<img>` unless decoding fails, then falls back to ImageBitmap.
   * - 'img' forces `<img>` and never calls `createImageBitmap`.
   */
  decodeMode?: ImageDecodeMode;
}

export interface TimelineOptions {
  /** Canvas width in px. @defaultValue 1000 */
  width?: number;
  /** Canvas height in px. @defaultValue 500 */
  height?: number;
  /** Canvas/backing-store configuration. */
  canvas?: {
    /** Device pixel ratio to use; 'auto' follows device. @defaultValue 'auto' */ dpr?: 'auto' | number;
    /**
     * Behaviour for dpr:'auto'.
     * - 'stable' (default) rounds devicePixelRatio up to the nearest integer.
     * - 'native' uses the raw devicePixelRatio value.
     * @defaultValue 'stable'
     */
    dprMode?: 'stable' | 'native';
    /** Maximum DPR to apply for performance. @defaultValue 3 */ maxDpr?: number;
  };
  /** Vertical offset of the main line in px. @defaultValue 40 */
  verticalOffset?: number;
  /** Enable user interactions (wheel/drag/pinch). @defaultValue true */
  enableUserControl?: boolean;
  /** Show interactive cursors. @defaultValue true */
  enableCursor?: boolean;
  /** Zoom configuration. */
  zoom?: TimelineZoomOptions;
  /** Initial date (top-left when position not offset). @defaultValue {year:1990,month:1,day:1} */
  initialDate?: DateParts;
  /** Shift BCE input dates by +1 so 1BCE = year 0 internally. @defaultValue false */
  shiftBceDates?: boolean;
  /**
   * Drag activation vicinity near the main line.
   * - boolean: enable/disable dragging anywhere on the timeline
   * - function: (event, relativePosition) => boolean; return true to allow dragging
   *   where relativePosition is { left: number; top: number }
   * - object: { up?: number; down?: number } in pixels from the main line
   * @defaultValue true
   */
  draggingVicinity?:
    | boolean
    | ((event: PointerEvent, relativePosition: { left: number; top: number }) => boolean)
    | { up?: number; down?: number };
  /** Canvas/UI styling. */
  style?: TimelineStyleOptions;
  /** Time band configuration. */
  timeBand?: TimeBandOptions;
  /** Article configuration. */
  article?: ArticleOptions;
  /** Configure image sanitization, caching, and loader behaviour. */
  image?: ImageOptions;
  /** Register event handlers via object literal: { 'article-click': fn, ... } */
  on?: Partial<EventHandlerMap>;
  /**
   * Called after every draw pass.
   *
   * @deprecated Deprecated since 1.3.0 — Use `on['timeline-render-end']` instead.
   */
  onRedraw?: (ctx: RedrawContext) => void;
  /**
   * Article click handler configured via this option.
   *
   * @deprecated Deprecated since 1.3.0 — Use `on['article-click']` for click events or `on['article-select']` for selection by any means.
   */
  onArticleClick?: (article: Article, event?: Event) => void;
  /**
   * Article double click handler configured via this option.
   *
   * @deprecated Deprecated since 1.3.0 — Use `on['article-dblclick']` instead.
   */
  onArticleDoubleClick?: (article: Article, event?: Event) => void;
  /**
   * Timeline save handler configured via this option.
   *
   * @deprecated Deprecated since 1.3.0 — Use `on['timeline-state-change']` instead.
   */
  onSave?: (state: string) => void;
}

// ------------------------------------------------------------------------------------
// Events
// ------------------------------------------------------------------------------------

export interface ZoomPayload { zoom: number; zoomDelta: number }
export interface WheelPayload { centre: { x: number; y: number }; zoom: number; zoomDelta: number }
export interface PinchPayload {
  centre: { x: number; y: number } | null;
  scale: number;
  zoom: number;
  distance: number;
  zoomDelta: number;
  dx?: number;
  totalZoomDelta: number;
  pointers: unknown[];
}
export interface ViewportDragPayload { startToken: unknown; offsetX: number; dx?: number; totalDx?: number }
export interface TimelineClickPayload { viewport: { x: number; y: number } }
export interface DragDelta { dx: number; dy: number; totalDx: number; totalDy: number }

/** Data delivered to redraw lifecycle handlers. */
export interface RedrawContext {
  canvasContext: CanvasRenderingContext2D;
  /** Present for timeline-render-end; omitted for timeline-render-start. */
  drawCycleContext?: unknown;
  top: number;
}

export interface EventPayloadMap {
  // zoom
  'zoom': [ZoomPayload];
  'zoom-wheel': [WheelPayload, WheelEvent];
  'zoom-pinch-start': [PinchPayload, PointerEvent];
  'zoom-pinch': [PinchPayload, PointerEvent];
  'zoom-pinch-end': [PinchPayload, PointerEvent];
  // viewport
  'viewport-drag-start': [ViewportDragPayload, PointerEvent];
  'viewport-drag': [ViewportDragPayload, PointerEvent];
  'viewport-drag-end': [ViewportDragPayload, PointerEvent];
  // articles
  'article-pointerenter': [Article, PointerEvent];
  'article-pointerleave': [Article, PointerEvent];
  'article-click': [Article, PointerEvent];
  'article-dblclick': [Article, PointerEvent];
  'article-select': [Article];
  'article-drag-start': [Article, PointerEvent];
  'article-drag': [Article, DragDelta, PointerEvent];
  'article-drag-end': [Article, PointerEvent];
  // timeline
  'timeline-render-start': [RedrawContext];
  'timeline-render-end': [RedrawContext];
  'timeline-click': [TimelineClickPayload, PointerEvent];
  'timeline-dblclick': [TimelineClickPayload, PointerEvent];
  'timeline-state-change': [string];
}

export type TimelineEvent = keyof EventPayloadMap;
export type EventHandlerMap = { [K in TimelineEvent]: (...args: EventPayloadMap[K]) => void };

// ------------------------------------------------------------------------------------
// Classes
// ------------------------------------------------------------------------------------

export interface AnimOptions {
  /** Duration in ms. @defaultValue 2000 */ duration?: number;
  /** Easing name. @defaultValue 'swing' */ easing?: EasingName;
  /** Callback when animation completes. */ complete?: () => void;
  /** Additional horizontal shift, used by pinch. */ offsetX?: number;
}

export class Timeline {
  /**
   * Create a timeline bound to an existing canvas element or a container element (a canvas will be created).
   * @example
   * const tl = new Timeline(containerEl, { initialDate: { year: 1066, month: 10, day: 14 } });
   */
  constructor(container: HTMLCanvasElement | HTMLElement, options?: TimelineOptions);

  // Data loading
  /** Load articles and redraw. */
  load(articles: ArticleData[]): void;
  /** Load time bands and redraw. */
  loadTimeBands(bands: TimeBandData[]): void;

  // Query
  getArticleById(id: string | number): Article | undefined;
  getActiveArticle(): Article | null;
  hasArticle(id: string | number): boolean;

  // Selection and ordering
  bringFront(articleId: string | number): void;
  /** Selects and brings the article to the front; emits 'article-select'. */
  select(articleId: string | number): void;
  /** Reorder articles to match the array of ids (topmost last). */
  reorderArticles(ids: Array<string | number>): void;

  // Navigation & view
  /** Set start date (left edge). String input accepted in "-YYYY-MM-DD" style. */
  setStartDate(date: Dmy | string, pixelOffset?: number): void;
  /** Set centre date using startDate + offset computation. */
  setCentreDate(date: Dmy | string, pixelOffset?: number): void;
  /** Current zoom value. */
  getZoom(): number;
  /** Set zoom (optionally pivoting around a pixel). */
  setZoom(zoom: number, pivotPixel?: number, offsetX?: number): void;
  /** Smooth pan by pixels (negative = right). */
  goToPixelAnim(pixels: number, options?: AnimOptions): void;
  /** Smoothly scroll so that the given date is centred/positioned. */
  goToDateAnim(dmy: Dmy, options?: AnimOptions): void;

  // Drawing
  /** Request a redraw (coalesced if an animation is running). */
  requestRedraw(redrawFunction?: () => void): void;
  /** Full draw cycle. */
  redraw(): void;
  /** Reposition-only draw (for zoom/start-date changes). */
  repositionRedraw(): void;
  /** Default draw including reposition and stacking. */
  defaultRedraw(): void;
  /**
   * @deprecated Deprecated since 1.3.0 — Use `timeline.on('timeline-render-end', handler)` instead.
   */
  onRedraw(handler: (ctx: RedrawContext) => void): void;

  // Interaction toggles
  enableZoomByWheel(): void;
  disableZoomByWheel(): void;

  // Misc utilities
  /** Current canvas width in CSS pixels. */
  getWidth(): number;
  /** Current applied device pixel ratio for the canvas. */
  getCanvasDpr(): number;
  /** Get canvas-relative coordinates for a DOM event. */
  getRelativePosition(evt: MouseEvent | PointerEvent | TouchEvent): { left: number; top: number };

  // Options
  /** Get or set options via dot-path or partial object. */
  setOption(path: string, value: unknown): void;
  setOption<T extends DeepPartial<TimelineOptions>>(patch: T): void;
  setOption(path: string): unknown;
  /** Resize the canvas. */
  setSize(width: number, height: number): void;

  // Event bus
  on<T extends TimelineEvent>(event: T, handler: (...args: EventPayloadMap[T]) => void): void;
  off<T extends TimelineEvent>(event: T, handler: (...args: EventPayloadMap[T]) => void): void;
}

export class Article {
  /** Update article data at runtime (dot-path or object). */
  setOption(path: string, value: unknown): void;
  setOption(patch: Partial<ArticleData>): void;
  setOption(path: string): unknown;

  /** Update normal style (dot-path or object). */
  setStyle(path: string, value: unknown): void;
  setStyle(style: Partial<ArticleStyle>): void;
  setStyle(path: string): unknown;

  /** Update hover style (dot-path or object). */
  setHoverStyle(path: string, value: unknown): void;
  setHoverStyle(style: Partial<ArticleStyle>): void;
  setHoverStyle(path: string): unknown;

  /** Update active style (dot-path or object). */
  setActiveStyle(path: string, value: unknown): void;
  setActiveStyle(style: Partial<ArticleStyle>): void;
  setActiveStyle(path: string): unknown;

  /** Move instantly to absolute canvas position. */
  moveTo(position: { left: number; top: number }): void;
  /** Move by delta from current position. */
  moveToOffset(offset: { left: number; top: number }): void;

  /** Measured card width. */
  getWidth(): number;
  /** Measured card height. */
  getHeight(): number;
}

export class TimeBand {
  /** Update band data at runtime (dot-path or object). */
  setOption(path: string, value: unknown): void;
  setOption(patch: Partial<TimeBandData>): void;
  setOption(path: string): unknown;

  /** Update band style (dot-path or object). */
  setStyle(path: string, value: unknown): void;
  setStyle(style: Partial<TimeBandStyle>): void;
  setStyle(path: string): unknown;
}

// ------------------------------------------------------------------------------------
// Default export for UMD / ESM default
// ------------------------------------------------------------------------------------

declare const _default: {
  Timeline: typeof Timeline;
  Article: typeof Article;
  Dmy: DmyConstructor;
  // Commonly used constants
  PRECISION_DAY: typeof PRECISION_DAY;
  PRECISION_MONTH: typeof PRECISION_MONTH;
  PRECISION_YEAR: typeof PRECISION_YEAR;
  PRECISION_DECADE: typeof PRECISION_DECADE;
  PRECISION_CENTURY: typeof PRECISION_CENTURY;
  PRECISION_MILLENNIUM: typeof PRECISION_MILLENNIUM;
  PRECISION_MILLION_YEARS: typeof PRECISION_MILLION_YEARS;
  PRECISION_BILLION_YEARS: typeof PRECISION_BILLION_YEARS;
  DENSITY_ALL: typeof DENSITY_ALL;
  DENSITY_LOW: typeof DENSITY_LOW;
  DENSITY_MEDIUM: typeof DENSITY_MEDIUM;
  DENSITY_HIGH: typeof DENSITY_HIGH;
  RANGE_ALL: typeof RANGE_ALL;
  RANGE_SCREEN: typeof RANGE_SCREEN;
};

export default _default;
