# HistropediaJS — Download Bundle

This bundle ships the pre-built HistropediaJS library plus two minimal demos: one for UMD script-tag usage and one for native ES modules. Full documentation lives at https://js.histropedia.com/documentation. This README covers only how to use the files in this download.

## Folder structure

dist/
  histropedia.esm.js
  histropedia.esm.d.ts
  histropedia.esm.min.js
  histropedia.esm.min.js.map

  histropedia.umd.js
  histropedia.umd.d.ts
  histropedia.umd.min.js
  histropedia.umd.min.js.map

demo/
  esm/
    demo.js
    index.html
  umd/
    demo.js
    index.html
  shared/
    demo.css

CHANGELOG.md
LICENSE.md
README.md   (this file)

## Which files to use

| Use case | File |
| --- | --- |
| Modern browsers, bundlers, native modules | dist/histropedia.esm.js (or dist/histropedia.esm.min.js) |
| Script tag / no build system | dist/histropedia.umd.js (or dist/histropedia.umd.min.js) |
| Type definitions | dist/histropedia.esm.d.ts and dist/histropedia.umd.d.ts |

The .min.js.map files are included for debugging minified builds.

## Compatibility

| Build | File | Target | Notes |
| --- | --- | --- | --- |
| ESM | dist/histropedia.esm.js | ES2020 | Best for bundlers and modern browsers. Uses native `async/await`, optional chaining (`?.`), and nullish coalescing (`??`). |
| UMD | dist/histropedia.umd.js | ES2017 | For `<script>` tags. Uses native `async/await`; optional chaining and nullish coalescing are transpiled down. |

Feature detection covers most optional Web APIs. The timeline still works without them, but performance may drop if `createImageBitmap`, `OffscreenCanvas`, `img.decode`, or `performance.now` are missing. Widely-supported APIs you need in very old environments: `fetch`, `Promise`, `Map`/`Set`, `URL`, and `requestAnimationFrame`.

## 1) Run the UMD demo (no build tools)

You can open `demo/umd/index.html` directly from your file system in any browser. It loads `dist/histropedia.umd.js` via a script tag and attaches the `Histropedia` global.

`demo/umd/demo.js` shows the script-tag setup:

```html
<div id="timeline-container"></div>

<script src="../../dist/histropedia.umd.js"></script>
<script src="./demo.js"></script>
```

```js
const { Timeline, PRECISION_MONTH, PRECISION_YEAR, PRECISION_DECADE, PRECISION_CENTURY } = Histropedia;

const container = document.getElementById("timeline-container");

const options = {
  width: container.clientWidth,
  height: container.clientHeight,
  initialDate: { year: 1650, month: 1, day: 1 },
  zoom: { initial: 47 },
};

const timeline = new Timeline(container, options);
timeline.load(articles);
```

Optional editor type checking on the global build:

```js
// @ts-check
/// <reference path="../../dist/histropedia.umd.d.ts" />
```

## 2) Run the ESM demo (native import)

Browsers require ES modules to be served over HTTP, not `file://`.

1. From the root of this bundle, start any static server:
   - `npx serve .`
   - or `python -m http.server 8000`
2. Open http://localhost:8000/demo/esm/

`demo/esm/demo.js` imports straight from the built ESM bundle:

```js
// @ts-check // optional; safe to remove

import {
  Timeline,
  PRECISION_MONTH,
  PRECISION_YEAR,
  PRECISION_DECADE,
  PRECISION_CENTURY,
} from "../../dist/histropedia.esm.js";

const container = document.getElementById("timeline-container");

/** @type {import("../../dist/histropedia.esm.d.ts").TimelineOptions} */
const options = {
  width: container.clientWidth,
  height: container.clientHeight,
  initialDate: { year: 1650, month: 1, day: 1 },
  zoom: { initial: 47 },
};

const timeline = new Timeline(container, options);
timeline.load(articles);
```

The demo keeps the canvas sized to the container on resize. You can remove the `@ts-check` line and JSDoc type hints if you prefer plain JavaScript.

## Using HistropediaJS in your project

Script-tag usage (no bundler):

```html
<script src="/path/to/histropedia.umd.min.js"></script>
<script>
  const { Timeline, PRECISION_YEAR } = Histropedia;

  const timeline = new Timeline(container, {
    initialDate: { year: 1990, month: 1, day: 1 },
    zoom: { initial: 34 },
  });

  timeline.load([
    {
      id: 1,
      title: "Example",
      from: { year: 2000, precision: PRECISION_YEAR },
    },
  ]);
</script>
```

ES modules (modern browsers and bundlers):

```js
import { Timeline, PRECISION_YEAR } from "./dist/histropedia.esm.js";

const timeline = new Timeline(container, {
  initialDate: { year: 1990, month: 1, day: 1 },
  zoom: { initial: 34 },
});
```

For API details, styling options, and advanced usage, see the online docs: https://js.histropedia.com/documentation.

## Licence

This bundle is distributed under the HistropediaJS licence. Review the terms at https://js.histropedia.com/licence. Commercial usage or redistribution may require additional permissions—contact us if unsure.
